/*
 * FormValidator.js
 *
 * Company: Hyfinity Ltd
 * Copyright (c) 2003
 *
 * Generic form validation class.
 * Allows the user to specify compulsory fields and other validation
 * information in a declarative manner, all form validation is then
 * taken care of automatically.
 *
 * @author <a href="rafel.bordas@hyfinity.com">Rafel Bordas</a>
 * @version 1.0
 */

 /*
  * Constructs a new form validator object.
  *
  * @param form A HTML form to validate
  */
function FormValidator(form) {
  this._form = form;
  //this._requiredFields = new Array();
  //this._invalidValues = new Array("", null);

  // Initialize the member function references
  // for the class prototype
  if (typeof(_FormValidatorPrototypeCalled) == 'undefined') {
    _FormValidatorPrototypeCalled = true;
    FormValidator.prototype.getForm = getForm;
    FormValidator.prototype.checkForm = checkForm;
    FormValidator.prototype.checkField = checkField;
    FormValidator.prototype.checkContainer = checkContainer;
    FormValidator.prototype.convertFormValues = convertFormValues;
  }

  //alert("created form validator");


  //set up the type validation classes
  this._numberValidator = new NumberValidator();
  this._stringValidator = new StringValidator();
  this._booleanValidator = new BooleanValidator();
  this._dateValidator = new DateValidator();


  /*
   * Return the HTML form associated with the validator
   *
   * @return The HTML Form being validated
   */
  function getForm() {
    return this._form;
  }



  /*
   * Validates the form object associated with this FormValidator.
   *
   * @return An array of ValidationError objects indicating the validation errors.
   */
  function checkForm()
  {
      //initalise a validation error object, so that the error code values will exist when needed
      new ValidationError(this._form.elements[0], 0);

      var invalidFields = new Array();
      for (var i = 0; i < this._form.elements.length; ++i)
      {
          var element = this._form.elements[i];

          var fieldErrors = this.checkField(element)
          if (fieldErrors != null)
          {
              invalidFields = invalidFields.concat(fieldErrors);
          }
      }

      return invalidFields;
  }

  /**
   * Checks the given field to see if it is valid.
   * @param field The field on the page to validate.
   * @return One or more ValidationError objects detailing the validation problems with the given field.
   *         If multiple errors, these are contained within an array, if no errors, null is returned.
   */
  function checkField(field)
  {
      if ((field.getAttribute("_type") != undefined) && (field.getAttribute("_type") != null) && (field.getAttribute("_type") != ''))
      {
          //check if the element should be validated
          if ((field.getAttribute("_validate") != 'false') && (!checkFieldHidden(field)))
          {
              switch(field.getAttribute("_type"))
              {
                  case "number":
                        return this._numberValidator.check(field);
                  case "string":
                        return this._stringValidator.check(field);
                  case "boolean":
                        return this._booleanValidator.check(field);
                  case "date":
                        return this._dateValidator.check(field);
                  default:
                        return new ValidationError(field, ValidationError.ERROR_INVALID_TYPE);
              }
          }
      }
      return null;
  }

  /**
   * Validates all fields within the given HTML container.
   * @param container The HTML container (eg DIV) containing the fields to validate.
   * @return An array of ValidationError objects indicating the valdiation errors with the fields checked.
   */
    function checkContainer(container)
    {
        if (container == null || typeof(container) == 'undefined')
            return null;

        var errors = new Array();

        //check any input fields
        var inputs = container.getElementsByTagName('input');
        for (var i = 0; i < inputs.length; ++i)
        {
            errors = errors.concat(this.checkField(inputs[i]));
        }

        //check any textareas
        inputs = container.getElementsByTagName('textarea');
        for (var i = 0; i < inputs.length; ++i)
        {
            errors = errors.concat(this.checkField(inputs[i]));
        }

        //check any select boxes
        inputs = container.getElementsByTagName('select');
        for (var i = 0; i < inputs.length; ++i)
        {
            errors = errors.concat(this.checkField(inputs[i]));
        }

        return errors;
  }

  /*
   * Checks each form element in the form associated with this formValidator
   * to see if its value needs to be converted.
   * If so this method actually changes the values of the fields on the form
   * This should be called before the form is submitted
   */
  function convertFormValues()
  {
      for (var i = 0; i < this._form.elements.length; ++i)
      {
          var element = this._form.elements[i];

          if ((element.getAttribute("_type") != undefined) && (element.getAttribute("_type") != null) && (element.getAttribute("_type") != ''))
          {
              //check if the element should be validated
              if (element.getAttribute("_validate") != 'false')
              {
                  ValueConverter.convertField(element)
              }
          }
      }
  }
}

/**
 * Returns a boolean value indicating whether or not the given field is currently hidden
 */
function checkFieldHidden(field)
{
    var disp = getCurrentStyle(field,"display");

    if (disp == 'none')
    {
        return true
    }
    else if (field.parentElement != null)
    {
        return checkFieldHidden(field.parentElement);
    }
    else
    {
        return false;
    }
}