/*
 * ErrorDisplay.js
 *
 * Company: Hyfinity Ltd
 * Copyright (c) 2003
 *
 * Generic javscript object for highlighting validation errors in a form
 *
 * @author Gerard Smyth
 * @version 1.0
 *
 */


function ErrorDisplay(displayMethod)
{
    this._displayMethod = displayMethod;
    this._showMessage = true;
    this._fieldStyle = "";
    this._messageStyle = "";
    this._fieldClass = "";
    this._messageClass = "";
    this._mainMessageLocation = "";
    this._mainMessageString = "";
    /** Stores whether popup alerts should be used. */
    this._showAlerts = false;
    /** Stores the message string that will be shown in the alert window.*/
    this._alertMessage = '';
    /** Indicates whether all errors should be highlighted ('all'), or just the first ('one').
     * This currently only applies to the alert messages. */
    this._validationMode = 'one';


    // provide backward compatability by handling the old display method of alert.
    if (displayMethod == 'alert')
    {
        this._displayMethod = 'none';
        this._showAlerts = true;
    }

    // Initialize the member function references
    // for the class prototype
    if (typeof(_ErrorDisplayPrototypeCalled) == 'undefined')
    {
        _ErrorDisplayPrototypeCalled = true;
        ErrorDisplay.prototype.showErrors = showErrors;
        ErrorDisplay.prototype.resetDisplay = resetDisplay;
        ErrorDisplay.prototype.setFieldStyle = setFieldStyle;
        ErrorDisplay.prototype.setMessageStyle = setMessageStyle;
        ErrorDisplay.prototype.setFieldClass = setFieldClass;
        ErrorDisplay.prototype.setMessageClass = setMessageClass;
        ErrorDisplay.prototype.setShowMessage = setShowMessage;
        ErrorDisplay.prototype.setMessageLocation = setMessageLocation;
        ErrorDisplay.prototype.setMessageString = setMessageString;
        ErrorDisplay.prototype.setShowAlerts = setShowAlerts;
        ErrorDisplay.prototype.setValidationMode = setValidationMode;
    }

    //set up private methods
    this.updateDisplay = updateDisplay;
    this.showErrorText = showErrorText;
    this.showErrorTooltip = showErrorTooltip;
    this.buildAlertMessage = buildAlertMessage;

    /** sets the css style to apply to a field container if it has an error*/
    function setFieldStyle(style)
    {
        this._fieldStyle = style;
    }
    /** sets the css class to apply to a field container if it has an error*/
    function setFieldClass(newClass)
    {
        this._fieldClass = newClass;
    }
    /** sets the css style to apply to a error message container if the associated field has an error*/
    function setMessageStyle(style)
    {
        this._messageStyle = style;
    }
    /** sets the css class to apply to a error message container if the associated field has an error*/
    function setMessageClass(newClass)
    {
        this._messageClass = newClass;
    }

    /** sets whether or not error messages should be shown with the highlighting*/
    function setShowMessage(showMessage)
    {
        this._showMessage = showMessage;
    }

    /** sets whether the error messages should be shown together in a top level DIV
     * @param messageLocation the name of the DIV to use to show the error messages, or null to stop this display*/
    function setMessageLocation(messageLocation)
    {
        this._mainMessageLocation = messageLocation;
    }
    /** sets a specific string that should be used in the top level DIV instead of the specific error messages*/
    function setMessageString(messageString)
    {
        this._mainMessageString = messageString;
    }

    /** sets whether alert messages shoulf be shown along with the visual highlighting.
     * Popup alert messages should be used for speech browsers. */
    function setShowAlerts(showAlerts)
    {
        this._showAlerts = showAlerts;
    }
    /** sets the validationMode. Either 'all' or 'one'.
     * This value is curretnly only used for the alert popup to decide whether to include all errors or just one.
     */
    function setValidationMode(validationMode)
    {
        this._validationMode = validationMode;
    }



    /*
     * Main function that takes an array of ValidationError objects
     * and calls updateDisplay method as needed to show the results.
     */
    function showErrors(validationErrors)
    {
        //sort the array
        //validationErrors.sort(sortErrors);
        var currentField = "";
        for (var i=0; i< validationErrors.length; ++i)
        {
            // check the field names don't end with _datefield_part_x
            var workingDisplayName = validationErrors[i].getField().name;
            var datefieldIndex = workingDisplayName.lastIndexOf("_datefield_part_");
            var endIsInteger = false;
            if(datefieldIndex != -1)
            {
                //now check there is an integer on the end
                var end_integer = workingDisplayName.substring(datefieldIndex+16,workingDisplayName.length);
                if(!isNaN(end_integer))
                {
                    endIsInteger = true;
                }
            }
            if((datefieldIndex != -1) && (endIsInteger))
            {
                //take off the datefield suffix
                validationErrors[i].getField().name = workingDisplayName.substring(0,datefieldIndex);
                //this._alertMessage += workingDisplayName.substring(0,datefieldIndex) + "\t:\t" + DisplayMessages.getMessage(error.getField(), error.getErrorCode()) + "\n";
            }

            if (validationErrors[i].getField().name == currentField)
                continue;

            if (validationErrors[i].getField().getAttribute("_element") == currentField)
                continue;

            if (validationErrors[i].getField().getAttribute("_element") != null)
                currentField = validationErrors[i].getField().getAttribute("_element");
            else
                currentField = validationErrors[i].getField().name;

            this.updateDisplay(validationErrors[i]);

        }

        //store the errors so they can be removed later
        this._shownErrors = validationErrors;

        if (validationErrors.length > 0)
        {
            if (this._showAlerts)
            {
                alert(this._alertMessage);
                this._alertMessage = "";
            }

            validationErrors[0].getField().focus();
        }
    }

    //validationError objects are sorted based on field name and then error code
    function sortErrors(e1, e2)
    {
        if(e1.getField().name > e2.getField().name)
            return -1
        if(e1.getField().name < e2.getField().name)
            return 1
        return e1.getErrorCode() - e2.getErrorCode();
    }

    /*
     * Function that takes a ValidationError object and displays it based on
     * this ErrorDisplay objects setting's
     */
    function updateDisplay(error)
    {
        if (this._displayMethod == 'text') //update the styling and contents of relevant containers to show error
        {
            this.showErrorText(error);
        }
        else if (this._displayMethod == 'tooltip') //update the styling of relevant containers and create tooltip to show error
        {
            this.showErrorTooltip(error);
        }
        else if (this._displayMethod == 'none')
        {
            //do nothing
        }
        else
        {
            alert('Unknown display method!!!');
        }

        //check if the errors should be shown in combined DIV
                if ((this._mainMessageLocation != undefined) && (this._mainMessageLocation != null) && (this._mainMessageLocation != ''))
        {
            this._mainMessageLocation.style.display = "block";
            if ((this._mainMessageString != undefined) && (this._mainMessageString != null) && (this._mainMessageString != ''))
                this._mainMessageLocation.innerHTML = this._mainMessageString;
            else
                this._mainMessageLocation.innerHTML = this._mainMessageLocation.innerHTML + "<br/>"+ getFieldDisplayName(error.getField())+": "+DisplayMessages.getMessage(error.getField(), error.getErrorCode());
        }

        if (this._showAlerts == true) //show alert messages for errors
        {
            this.buildAlertMessage(error);
        }
    }

    /** Build the message to show in a popup alert.
     * If the validationMode is set to 'one' then only the first error will be alerted,
     * otherwise the alert message will detail all errors.
     */
    function buildAlertMessage(error)
    {
        if ((this._validationMode == 'one') && (this._alertMessage == ''))
        {
           this._alertMessage = "Validation Error on " + getFieldDisplayName(error.getField()) +
                        "\n\n" + DisplayMessages.getMessage(error.getField(), error.getErrorCode());
        }

        if (this._validationMode == 'all')
        {
            if (this._alertMessage == '')
            {
                this._alertMessage = 'Please correct the following errors:\n\n';
            }

                        //check the message for datefield parts to be renamed.
                        var workingDisplayName = getFieldDisplayName(error.getField());
                        var datefieldIndex = workingDisplayName.lastIndexOf("_datefield_part_");
                        var endIsInteger = false;
                        if(datefieldIndex != -1)
                        {
                            //now check there is an integer on the end
                            var end_integer = workingDisplayName.substring(datefieldIndex+16,workingDisplayName.length);
                            if(!isNaN(parseInt(end_integer)))
                            {
                                endIsInteger = true;
                            }
                        }
                        if((datefieldIndex != -1) && (endIsInteger))
                        {
                            //take off the datefield suffix
                            this._alertMessage += workingDisplayName.substring(0,datefieldIndex) + "\t:\t" + DisplayMessages.getMessage(error.getField(), error.getErrorCode()) + "\n";
                        }
                        else
                        {
                            this._alertMessage += getFieldDisplayName(error.getField()) + "\t:\t" + DisplayMessages.getMessage(error.getField(), error.getErrorCode()) + "\n";
                        }
        }
    }

    /** Shows the given ValidationError object using the 'text' style display
     *  ie. changes styles of the field and places the error message within the message container*/
    function showErrorText(error)
    {
        var fieldName = error.getField().name;
        //check if the field is only used for display, and if so remove the '_display' suffix to find the field containers
        if (error.getField().getAttribute("_display_only") == 'true')
        {
            //alert("field "+fieldName+" ends with '_display'");
            fieldName = fieldName.substring(0, fieldName.length - 8);
            //alert("new field name: "+fieldName);
        }

        /** if this is for a selectMany control, the name of the field will have an extra number at the end
         *  Therefore use the value of the _element attribute instead
         */
        if (error.getField().getAttribute("_element") != null)
            fieldName = error.getField().getAttribute("_element");

        var fieldContainer = document.getElementById(fieldName+"_container");
        var messageContainer = document.getElementById(fieldName+"_message_container");
        if (fieldContainer != undefined)
        {
            //Store the current class and style settings for the field container
            if ((fieldContainer.className != undefined) && (fieldContainer.className != ""))
                fieldContainer.oldClass=fieldContainer.className;
            else
                fieldContainer.oldClass="";
            fieldContainer.oldStyle = fieldContainer.style.cssText;

            //set the error styles for the field container
            if (this._fieldStyle != '')
            {
                if (is_ie5up)
                {
                    fieldContainer.style.cssText = this._fieldStyle;
                }

                else /* if (is_nav6up) */
                {
                    fieldContainer.setAttribute("style", this._fieldStyle);
                }
            }
            if (this._fieldClass != '')
                fieldContainer.className = this._fieldClass;
        }

        if (messageContainer != undefined)
        {
            //Store the current class and style settings for the message container
            /* if ((messageContainer.className != undefined) && (messageContainer.className != ""))
                messageContainer.oldClass=messageContainer.className;
            else
                messageContainer.oldClass = "";
            messageContainer.oldStyle = messageContainer.style.cssText; */

            //set the error styles for the message container
            if (this._messageStyle != '')
            {
                if (is_ie5up)
                {
                    messageContainer.style.cssText = this._messageStyle;
                }
                else /* if (is_nav6up) */
                {
                    messageContainer.setAttribute("style", this._messageStyle);
                }
            }
            if (this._messageClass != '')
                messageContainer.className = this._messageClass;
            messageContainer.innerHTML="&nbsp;";
        }

        if ((this._showMessage) && (messageContainer != undefined))
        {
            // alert("show hint: "+messageContainer.getAttribute("_hintMessageShow"));
            if ((messageContainer.getAttribute("_hintMessageShow") != null) && (messageContainer.getAttribute("_hintMessageShow") == 'true'))
            {
                messageContainer.innerHTML = messageContainer.getAttribute("_hintMessage");
            }
            else
            {
                messageContainer.innerHTML = DisplayMessages.getMessage(error.getField(), error.getErrorCode());
            }

        }



    }

    /** Shows the given ValidationError object using the 'tooltip' style display
     *  ie. changes styles of the field and places the error message within an onhover tooltip*/
    function showErrorTooltip(error)
    {
        var fieldName = error.getField().name;

        //check if the field is only used for display, and if so remove the '_display' suffix to find the field containers
        if (error.getField().getAttribute("_display_only") == 'true')
        {
            //alert("field "+fieldName+" ends with '_display'");
            fieldName = fieldName.substring(0, fieldName.length - 8);
            //alert("new field name: "+fieldName);
        }

        /** if this is for a selectMany control, the name of the field will have an extra number at the end
         *  Therefore use the value of the _element attribute instead
         */
        if (error.getField().getAttribute("_element") != null)
            fieldName = error.getField().getAttribute("_element");

        var fieldContainer = document.getElementById(fieldName+"_container");
        var messageContainer = document.getElementById(fieldName+"_message_container");
        if (fieldContainer != undefined)
        {
            // alert("about to update field container styling");
            //Store the current class and style settings for the field container
            if ((fieldContainer.className != undefined) && (fieldContainer.className != ""))
                fieldContainer.setAttribute("oldClass", fieldContainer.className);
            else
                fieldContainer.setAttribute("oldClass", "");

            fieldContainer.setAttribute("oldStyle", fieldContainer.style.cssText);

            //set the error styles for the field container
            if (this._fieldStyle != '')
            {
                if (is_ie5up) //use the cssText property to set the css string
                {
                    fieldContainer.style.cssText = this._fieldStyle;
                }
                else //try and use the setAttribute method to set the css string
                {
                    fieldContainer.setAttribute("style", this._fieldStyle);
                }
            }
            if (this._fieldClass != '')
                fieldContainer.className = this._fieldClass;
        }

        if (messageContainer != undefined)
        {
            // alert("about to update message container styling");
            //Store the current class and style settings for the message container
            /* if ((messageContainer.className != undefined) && (messageContainer.className != ""))
                messageContainer.setAttribute("oldClass", messageContainer.className);
            else
                messageContainer.setAttribute("oldClass", "");

            messageContainer.setAttribute("oldStyle", messageContainer.style.cssText); */

            //set the error styles for the message container
            if (this._messageStyle != '')
            {
                if (is_ie5up) //use the cssText property to set the css string
                {
                    messageContainer.style.cssText = this._messageStyle;
                }
                else /* if (is_nav6up) */ //try and use the setAttribute method to set the css string
                {
                    messageContainer.setAttribute("style", this._messageStyle);
                }
            }
            if (this._messageClass != '')
                messageContainer.className = this._messageClass;

            /**@todo need an alternative for this hack to ensure the pictures set using css background are shown in netscape*/
            messageContainer.innerHTML="&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";
        }

        if (this._showMessage && messageContainer != undefined)
        {
            // alert("show hint: "+messageContainer.getAttribute("_hintMessageShow"));
            if ((messageContainer.getAttribute("_hintMessageShow") != null) && (messageContainer.getAttribute("_hintMessageShow") == 'true'))
            {
                messageContainer.setAttribute("_tipMessage", messageContainer.getAttribute("_hintMessage"));
            }
            else
            {
                messageContainer.setAttribute("_tipMessage", DisplayMessages.getMessage(error.getField(), error.getErrorCode()));
            }
            messageContainer.onmousemove = showTipMessage;
            messageContainer.onmouseout = hideTipMessage;
        }

    }


    /** Resets the display by undoing any changes made by showing errors on the form */
    function resetDisplay()
    {
        // alert("about to reset display");
        var elements = this._shownErrors

        if ((elements != undefined) && ((this._displayMethod == 'text') || (this._displayMethod == 'tooltip')))
        {
            for (var i=0; i< elements.length; ++i)
            {
                var fieldName = elements[i].getField().name;
                //check if the field is only used for display, and if so remove the '_display' suffix to find the field containers
                if (elements[i].getField().getAttribute("_display_only") == 'true')
                {
                    fieldName = fieldName.substring(0, fieldName.length - 8);
                }

                /* if this is for a selectMany control, the name of the field will have an extra number at the end
                 *  Therefore use the value of the _element attribute instead*/
                if (elements[i].getField().getAttribute("_element") != null)
                    fieldName = elements[i].getField().getAttribute("_element");

                var fieldContainer = document.getElementById(fieldName+"_container");
                var messageContainer = document.getElementById(fieldName+"_message_container")


                if (fieldContainer != undefined)
                {
                    if (fieldContainer.getAttribute("oldClass") != null)
                        fieldContainer.className = fieldContainer.getAttribute("oldClass");

                    if (fieldContainer.getAttribute("oldStyle") != null)
                    {
                        if (is_ie5up)
                        {
                            fieldContainer.style.cssText = fieldContainer.getAttribute("oldStyle");
                        }
                        else
                        {
                            fieldContainer.setAttribute("style", fieldContainer.getAttribute("oldStyle"));
                        }
                    }
                }
                if (messageContainer != undefined)
                {
                    //alert("resetting message container styling");
                    //check if the hint or server message should be reshown
                    var hintMessageShow = messageContainer.getAttribute("_hintMessageShow");
                    var serverMessageShow = messageContainer.getAttribute("_serverMessageShow");
                    var serverMessageRetain = messageContainer.getAttribute("_serverMessageRetain");
                    //alert("show hint: "+hintMessageShow+" show server: "+serverMessageShow+" retain server: "+serverMessageRetain);
                    if ((serverMessageShow == 'true') && (serverMessageRetain == 'true'))
                    {
                        //alert("server message should be reshown");
                        //reshow server message
                        if (messageContainer.getAttribute("_serverClass") != null)
                            messageContainer.className = messageContainer.getAttribute("_serverClass");

                        if (messageContainer.getAttribute("_serverStyle") != null)
                        {
                            if (is_ie5up)
                            {
                                messageContainer.style.cssText = messageContainer.getAttribute("_serverStyle");
                            }
                            else
                            {
                                messageContainer.setAttribute("style", messageContainer.getAttribute("_serverStyle"));
                            }
                        }
                        if (this._displayMethod == 'tooltip')
                        {
                            messageContainer.setAttribute("_tipMessage", messageContainer.getAttribute("_serverMessage"));
                            messageContainer.innerHTML = "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";
                            messageContainer.onmousemove = showTipMessage;
                            messageContainer.onmouseout = hideTipMessage;
                        }
                        else if (this._displayMethod == 'text')
                        {
                            messageContainer.innerHTML = messageContainer.getAttribute("_serverMessage");
                            messageContainer.setAttribute("_tipMessage", "");
                            messageContainer.onmousemove = null;
                            messageContainer.onmouseout = null;
                        }
                        else
                        {
                            messageContainer.innerHTML = "&nbsp;"
                            messageContainer.setAttribute("_tipMessage", "");
                            messageContainer.onmousemove = null;
                            messageContainer.onmouseout = null;
                        }
                    }
                    else if (hintMessageShow == 'true')
                    {
                        // alert("hint message should be reshown");
                        //reshow hint message
                        if (messageContainer.getAttribute("_hintClass") != null)
                            messageContainer.className = messageContainer.getAttribute("_hintClass");

                        if (messageContainer.getAttribute("_hintStyle") != null)
                        {
                            if (is_ie5up)
                            {
                                messageContainer.style.cssText = messageContainer.getAttribute("_hintStyle");
                            }
                            else
                            {
                                messageContainer.setAttribute("style", messageContainer.getAttribute("_hintStyle"));
                            }
                        }
                        if (this._displayMethod == 'tooltip')
                        {
                            messageContainer.setAttribute("_tipMessage", messageContainer.getAttribute("_hintMessage"));
                            messageContainer.innerHTML = "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";
                            messageContainer.onmousemove = showTipMessage;
                            messageContainer.onmouseout = hideTipMessage;
                        }
                        else if (this._displayMethod == 'text')
                        {
                            messageContainer.innerHTML = messageContainer.getAttribute("_hintMessage");
                            messageContainer.setAttribute("_tipMessage", "");
                            messageContainer.onmousemove = null;
                            messageContainer.onmouseout = null;
                        }
                        else
                        {
                            messageContainer.innerHTML = "&nbsp;"
                            messageContainer.setAttribute("_tipMessage", "");
                            messageContainer.onmousemove = null;
                            messageContainer.onmouseout = null;
                        }
                    }
                    else
                    {
                        // alert("styling should be cleared");
                        //clear all styling
                        messageContainer.className = "";
                        if (is_ie5up)
                            messageContainer.style.cssText = "";
                        else
                            messageContainer.setAttribute("style", "");

                        messageContainer.innerHTML = "&nbsp;"
                        messageContainer._tipMessage = "";
                        messageContainer.onmousemove = null;
                        messageContainer.onmouseout = null;
                    }
                }
            }
        }

        if ((this._mainMessageLocation != undefined) && (this._mainMessageLocation != null) && (this._mainMessageLocation != ''))
        {
            this._mainMessageLocation.style.display = "none";
            this._mainMessageLocation.innerHTML = "";
        }
    }
}



function showTipMessage(e)
{
    var tipDiv = document.getElementById('tipDiv');
    var messageDiv = document.getElementById("messageDiv")
    if ((tipDiv != undefined) && (messageDiv != undefined))
    {
        var posx = 0;
        var posy = 0;
        var evt;
        var target;
        if (!e)
        {
            evt = window.event;
            target = evt.srcElement;
        }
        else //netscape
        {
            evt = e;
            target = evt.currentTarget;
        }

        //get the mouse coords
        if (evt.pageX || evt.pageY)
        {
            posx = evt.pageX;
            posy = evt.pageY;
        }
        else if (evt.clientX || evt.clientY)
        {
            posx = evt.clientX
            posy = evt.clientY
            if ((document.body) && (document.body.scrollLeft || document.body.scrollTop))
            {
                //IE 4,5,and 6 (Non standards compliant)
                posx += document.body.scrollLeft;
                posy += document.body.scrollTop;
            }
            else if ((document.documentElement) &&
                     ((document.documentElement.scrollLeft) ||
                     (document.documentElement.scrollTop)))
            {
                //IE 6 (Standards compliant mode)
                posx += document.documentElement.scrollLeft;
                posy += document.documentElement.scrollTop;
            }
        }


        if (target != undefined)
        {
            if ((target.getAttribute("_tipMessage") != undefined) && (target.getAttribute("_tipMessage") != null)&& (target.getAttribute("_tipMessage") != ''))
            {
                messageDiv.innerHTML = target.getAttribute("_tipMessage");

                if (is_ie) //IE
                {
                    tipDiv.style.top = posy + Number(10);
                    var leftPos = Number(posx) - Number(tipDiv.style.pixelWidth);
                    if (leftPos < 5)
                        tipDiv.style.left = 5;
                    else
                        tipDiv.style.left = leftPos;

                    var messageHeight = messageDiv.clientHeight  + Number(4);
                    tipDiv.style.pixelHeight = messageHeight;
                    document.all.tipFrame.style.pixelHeight = messageHeight;
                }
                else /*if(is_nav6up)*/ //netscape
                {
                    //Gecko browsers work best if all positions/sizes etc are specified as a string with
                    //the units suffix (ie 'px' in this case)

                    tipDiv.style.top = (posy + Number(10)) + "px";
                    var leftPos = Number(posx) - tipDiv.style.width.substr(0, tipDiv.style.width.length-2);
                    if (leftPos < 5)
                        tipDiv.style.left = "5px";
                    else
                        tipDiv.style.left = leftPos + "px";

                    messageDiv.style.position = "fixed";

                    messageDiv.style.top = (posy + Number(10) - window.pageYOffset) + "px";;

                    leftPos = (posx - window.pageXOffset) - tipDiv.style.width.substr(0, tipDiv.style.width.length-2);
                    if (leftPos < 5)
                        messageDiv.style.left = "5px";
                    else
                        messageDiv.style.left = leftPos + "px";

                    messageDiv.style.width = tipDiv.style.width;

                    var messageHeight = document.defaultView.getComputedStyle(messageDiv, '').getPropertyValue("height");
                    tipDiv.style.height = messageHeight + "px";

                    document.getElementById("tipFrame").style.height = (messageHeight.substr(0, messageHeight.length-2) - Number(4)) + "px";
                }
                //QUESTION: What about other browsers, eg Opera

                tipDiv.style.visibility = 'visible';
                tipDiv.style.display = "block";
                messageDiv.style.visibility = "visible";
                messageDiv.style.display = "block";

                return;
            }
        }
        tipDiv.style.visibility = "hidden";
        tipDiv.style.display = "none";
        messageDiv.style.visibility = "hidden";
        messageDiv.style.display = "none";
    }
}

function hideTipMessage(e)
{
    var tipDiv = document.getElementById('tipDiv');
    var messageDiv = document.getElementById("messageDiv")
    if (tipDiv != undefined)
    {
        tipDiv.style.visibility = "hidden";
        tipDiv.style.display = "none";
    }
    if (messageDiv != undefined)
    {
        messageDiv.style.visibility = "hidden";
        messageDiv.style.display = "none";
    }
}

/** Tries to find the display name to use for the given field.
 * This is done by trying to find a label that is associated with this field,
 * and if present returning its content.
 * If this cant be found, the name of the field is simply returned.
 *
 * @param field The form control object to find the name for
 * @return the display name to use for the given field.
 */
function getFieldDisplayName(field)
{
    var id = field.getAttribute("id");

    if (field.type == 'radio')
        id = field.name + "1";

    //TODO: '_use' and '_element' attributes dont make it to the browser any more so
    // this wont work!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    if ((field.type == 'checkbox') && (field.getAttribute("_use") == 'selectMany'))
        id = field.getAttribute("_element") + "1";

    var usefieldset = false;
    var fieldset = null;
    //Handle the case where the radio buttons / checkboxes are contained in a fieldset,
    //but without a specific label.  In this case, we should use the fieldset legend.
    if ((field.type == 'radio') || ((field.type == 'checkbox') && (field.getAttribute("_use") == 'selectMany')))
    {
        fieldset = findFieldsetParent(field);
        if (fieldset != null)
        {
            usefieldset = true;
            inputs = fieldset.getElementsByTagName("input");
            selects = fieldset.getElementsByTagName("select");
            textareas = fieldset.getElementsByTagName("textarea");
            /*alert("inputs: " + inputs.length + ", selects: " + selects.length + ", textareas: " + textareas.length);*/
            for (var i = 0; i < inputs.length; ++i)
            {
                inputField = inputs.item(i);

                if (inputField.type == 'radio')
                {
                    if (inputField.name != field.name)
                        usefieldset = false;
                }
                else if (inputField.type == 'checkbox')
                {
                    /*alert(inputField.getAttribute("_element"));
                    alert(field.getAttribute("_element"));
                    alert(inputField.getAttribute("_element") != field.getAttribute("_element"));*/
                    if (inputField.getAttribute("_element") != field.getAttribute("_element"))
                        usefieldset = false;
                }
                else if (inputField.type == 'hidden')
                {
                    /*if (inputField.name.substr(inputField.name.length - 6) != "_xpath")
                        usefieldset = false;*/
                }
                else
                    usefieldset = false;
            }
            if ((selects.length > 0) || (textareas.length > 0))
            {
                usefieldset = false;
            }
        }
    }

    if (usefieldset)
    {
        if (fieldset != null)
        {
            return fieldset.getElementsByTagName("legend").item(0).innerHTML;
        }
        else
            return field.name;
    }
    else
    {
        if (id != null)
        {
            var labels = document.getElementsByTagName("label");
            for (var i = 0; i < labels.length; ++i)
            {
                var forAtt = labels.item(i).getAttribute('htmlFor');
                if (forAtt == null)
                    forAtt = labels.item(i).getAttribute('for');
                if (forAtt == id)
                {
                    var labelHTML = labels.item(i).innerHTML;
                    var startSpan = labelHTML.indexOf('<SPAN');
                    if (startSpan == -1)
                        startSpan = labelHTML.indexOf('<span');
                    if (startSpan != -1)
                    {
                        if (startSpan == 0)
                        {
                            var endSpan = labelHTML.indexOf('</SPAN>');
                            if (endSpan == -1)
                                endSpan = labelHTML.indexOf('</span>');
                            return labelHTML.substr(endSpan + 7);
                        }
                        else
                        {
                            return labelHTML.substring(0, startSpan);
                        }
                    }
                    else
                        return labelHTML;
                }
            }
        }
    }
    return field.name;
}


function findFieldsetParent(node)
{
    if (node.parentNode == null)
    {
        return null;
    }
    else
    {
        if (node.parentNode.tagName == 'FIELDSET')
        {
            return node.parentNode;
        }
        else
        {
            return findFieldsetParent(node.parentNode);
        }
    }
}

