/* ===================================================================================================
* WARNING – This file is part of the base implementation for WebMaker, so it should not be edited or changed for any project. 
* These files are replaced if a project is re-imported to the WebMaker Studio or migrated to a new version of the product. 
* For guidance on ‘How do I override or clone Hyfinity webapp files such as CSS & javascript?’, please read the following relevant FAQ entry: 
* http://www.hyfinity.net/faq/index.php?solution_id=1113
==================================================================================================== */

/**
 * DisplayMessages.js
 *
 * Company: Hyfinity Ltd
 * Copyright (c) 2003
 *
 * Class that generates display messages for a given error code
 * Will need to be replaced to create custom messages.
 *
 * @author Gerard Smyth
 * @version 1.1
 *
 */

hyf.validation.DisplayMessages = {
    messageProducers : {}
}

/**
 * Adds a message producer function that should be used to try and determine the error message
 * to display for the particular type of error.
 * When a message is being requested of this type, then each producer function added will be called
 * in the order they were added.  The first one that returns a message string will be used as the message
 * to be displayed.  When called, each function will be passed 2 parameters, the HTML field that the error
 * message should be found for, and the error code indicating the type of error
 * Producer functions for a specific error code will be called before any generic message producers.
 * If none of the producer functions defined return a message, then the default functionality will be used.
 * @param func A reference to the message producer function.
 * @param errorCode The code value indicating the type of errors that this message producer can deal with.
 *                  If this is not provided, then the function will be associated with all types of error.
 */
hyf.validation.DisplayMessages.addMessageProducer = function(func, errorCode)
{
    if ((errorCode == null) || (typeof(errorCode) == 'undefined'))
        errorCode = 'ALL';

    if (typeof(hyf.validation.DisplayMessages.messageProducers[errorCode]) == 'undefined')
        hyf.validation.DisplayMessages.messageProducers[errorCode] = new Array();

    hyf.validation.DisplayMessages.messageProducers[errorCode].push(func);
}

/**
 * Get the message associated with the test that this field has failed
 * This will call any defined message producer functions as needed to find the relevant message to display.
 * @param field The form object (input control, textarea, etc) that has failed the validation
 * @param errorCode The code number indicating the type of field - see validationError
 *
 * @return the message that should be displayed for this error.
 */
hyf.validation.DisplayMessages.getMessage = function(field, errorCode)
{
    //first check for any specifc message producers
    if (typeof(hyf.validation.DisplayMessages.messageProducers[errorCode]) != 'undefined')
    {
        for(var i = 0; i < hyf.validation.DisplayMessages.messageProducers[errorCode].length; ++i)
        {
            var func = hyf.validation.DisplayMessages.messageProducers[errorCode][i];
            var msg = func(field, errorCode);
            if ((msg != null) && (typeof(msg) == 'string'))
                return msg;
        }
    }

    //now check for any generic message producers
    if (typeof(hyf.validation.DisplayMessages.messageProducers['ALL']) != 'undefined')
    {
        for(var i = 0; i < hyf.validation.DisplayMessages.messageProducers['ALL'].length; ++i)
        {
            var func = hyf.validation.DisplayMessages.messageProducers['ALL'][i];
            var msg = func(field, errorCode);
            if ((msg != null) && (typeof(msg) == 'string'))
                return msg;
        }
    }

    //none of the defined message producers want to give a messge for this error, so use the default functionality
    return hyf.validation.DisplayMessages.getDefaultMessage(field, errorCode);
}

/**
 * Returns the display message that would be given for the deefined error if no message producers override it.
 * @param field The form object (input control, textarea, etc) that has failed the validation
 * @param errorCode The code number indicating the type of field - see validationError
 * @return the default message that should be displayed for this error.
 */
hyf.validation.DisplayMessages.getDefaultMessage = function(field, errorCode)
{
    switch (errorCode)
    {
        case hyf.validation.ValidationError.ERROR_INVALID_TYPE : return "Type attribute is invalid.";
        case hyf.validation.ValidationError.ERROR_REQUIRED : return "You must specify a value for this field.";
        case hyf.validation.ValidationError.ERROR_NAN : return "This value must represent a valid number.";
        case hyf.validation.ValidationError.ERROR_INVALID_DATE : if (field.getAttribute("_error_data_format_display") && (field.getAttribute("_error_data_format_display") != ''))
                                                                     return "This value must be a valid date in the format "+field.getAttribute("_error_data_format_display")+".";
                                                                 else if (field.getAttribute("_display_date_format") && (field.getAttribute("_display_date_format") != ''))
                                                                     return "This value must be a valid date in the format "+field.getAttribute("_display_date_format")+".";
                                                                 else
                                                                     return "This value must be a valid date in the format "+field.getAttribute("_data_date_format")+".";
        case hyf.validation.ValidationError.ERROR_MAX_INCLUSIVE : return "This value cannot be greater than "+field.getAttribute("_maxInclusive")+".";
        case hyf.validation.ValidationError.ERROR_MAX_EXCLUSIVE : return "This value must be less than "+field.getAttribute("_maxExclusive")+".";
        case hyf.validation.ValidationError.ERROR_MIN_INCLUSIVE : return "This value cannot be less than "+field.getAttribute("_minInclusive")+".";
        case hyf.validation.ValidationError.ERROR_MIN_EXCLUSIVE : return "This value must be greater than "+field.getAttribute("_minExclusive")+".";
        case hyf.validation.ValidationError.ERROR_REGULAR_EXPRESSION : return "This value must match the regular expressions "+field.getAttribute("_regularExpression").split('*@*@*')+".";
        case hyf.validation.ValidationError.ERROR_LENGTH : return "This value must be of length "+field.getAttribute("_length")+".";
        case hyf.validation.ValidationError.ERROR_MIN_LENGTH : return "This value must have a minimum length of "+field.getAttribute("_minLength")+".";
        case hyf.validation.ValidationError.ERROR_MAX_LENGTH : return "This value cannot have a length greater than "+field.getAttribute("_maxLength")+".";
        case hyf.validation.ValidationError.ERROR_NOT_VALID_BOOLEAN : return "This field can only have a value of 'true' or 'false'.";
        default: return "This field is invalid.";
    }
}




